function [ ] = LUAM0_map_setup( )
%
% Land Use Allocation Module - function 0 (LUAM0)
%
% This file loads the raw raster maps, calculates land use pixel
% probabilities and vectorizes the maps. The vectorized maps are then saved
% in excel and matlab-data formats.
%
% This function is not run during the execution of the allocation module
% (or LURNZ). It is run once to setup the vectorized map that LURNZ takes
% as input. If and only if the raw / underlying data is updated should this
% script be rerun.
%

% Code by Simon Anastasiadis : 2011-11-03
% Revised by Simon Anastasiadis: 2013-04-08

%% Parameters

% Resolution
% This depends on the resolution of the underlying maps, currently only two
% resolutions are available: 25ha per pixel and 1ha per pixel.
% resolution = 25;
resolution = 25;

% output path
output_path = '..\Data for Modules\Data 2 Land Use Allocation Module\';

% number of maps
num_maps = 11;

%% Setup path

if resolution == 25
    load_folder = 'Raster ASCI - 25 ha';
elseif resolution == 1
    load_folder = 'Raster ASCI - 1 ha';
end

% shared load path
map_path = '..\..\..\Data\';
% specific load path
map_path = [map_path, load_folder];

%% Set map names & paths

% base map / starting map
% base_map = [map_path,'\','LandUseMap_Year_2002.asc'];
base_map = [map_path,'\','LandUseMap_Year_2008.asc'];

% Land Use Capability map (LUC)
luc_map = [map_path,'\','LUC_NZ.asc'];
% Pasture Productivity map (PProd)
% pprod_map = [map_path,'\','pprod_asc.asc'];
% Slope map
slope_map = [map_path,'\','slope_asc.asc'];
% Maori ownership map
maori_map = [map_path,'\','maori_asc.asc'];
% Distance to ports
dist_port_map = [map_path,'\','dist_ports_asc.asc'];
% Distance to supermarkets (proxy for distance to town)
dist_super_map = [map_path,'\','dist_super.asc'];

% Regional council map
rc_map = [map_path,'\','rc_asc.asc'];
% Territorial authority map
ta_map = [map_path,'\','ta_asc.asc'];

% LUCAS map (proxy for forest age class
LUCAS_map = [map_path,'\','lucas_asc.asc'];

% Forestare age map
ForestAge_map = [map_path,'\','Forestry_Agemap2008.asc'];

%% Load base_map into Matlab

% inform user
msg = sprintf(' - - LUCM0 : loading base map');
disp(msg)

% loadASCI map files
% map = importdata( PATH , DELIMITER = SPACE , HEADER_LINES = 6 )
base_map  = importdata(base_map,' ',6);

% inform user
msg = sprintf(' - - LUCM0 : base map loaded');
disp(msg)

%% Construct GIS header information

GISheader = ConstructGISHeader(base_map.textdata, output_path, resolution );

%% Load, Combine and vectorize all maps - setup
% we process and clear each map one at a time to save space
% (at 1 ha resolution loading 10 maps fills >90% of available memory and
% Matlab does not have enough space to process the maps).

% inform user
msg = sprintf(' - - LUCM0 : loading & combining all other maps');
disp(msg)

% establish LURNZGISDATA
ncol = GISheader(1);
nrow = GISheader(2);
LURNZDATA = zeros( nrow * ncol ,num_maps+1);

% column tracker
c.sorter = 1;
c.base_map = 2;
c.luc = 3;
c.pprod = 4;
c.slope = 5;
c.maori = 6;
c.dist_port = 7;
c.dist_super = 8;
c.rc = 9;
c.ta = 10;
c.LUCAS = 11;
c.forestAge = 12;

% fill sorter and base map column
LURNZDATA(:,c.sorter)   = (1:nrow*ncol).';
LURNZDATA(:,c.base_map) = reshape(base_map.data,  nrow*ncol, 1);

% indicator for land pixels
sea_pixel = GISheader(6);
iNonSea = LURNZDATA(:,c.base_map) ~= sea_pixel;

% trim sorter and base map in LURNZ DATA
LURNZDATA = LURNZDATA(iNonSea,:);

%% Load, Combine and vectorize all maps - map work
% for each map:
% - load map
% - use iNonSea to extract land pixels
% - save land pixels to LURNZDATA
% - clear map to save space

% note
% iNonSea and LURNZDATA(:,c.sorter) are both indicators of the land pixels
% map.data(iNonSea) and map.data(LURNZDATA(:,c.sorter))
% are equivalent as both extract from map.data the land pixels
% we use iNonSea in this code

% map = importdata( PATH , DELIMITER = SPACE , HEADER_LINES = 6 )
% LURNZDATA(:,c.map) = map.data(iNonSea);
% clear('map')

luc_map   = importdata(luc_map , ' ', 6);
LURNZDATA(:,c.luc)   = luc_map.data(iNonSea);
clear('luc_map')

% pprod_map = importdata(pprod_map, ' ', 6);
% LURNZDATA(:,c.pprod) = pprod_map.data(iNonSea);
% clear('pprod_map')

slope_map = importdata(slope_map, ' ', 6);
LURNZDATA(:,c.slope) = slope_map.data(iNonSea);
clear('slope_map')

maori_map = importdata(maori_map, ' ', 6);
LURNZDATA(:,c.maori) = maori_map.data(iNonSea);
clear('maori_map')

dist_port_map  = importdata(dist_port_map, ' ', 6);
LURNZDATA(:,c.dist_port)  = dist_port_map.data(iNonSea);
clear('dist_port_map')

dist_super_map = importdata(dist_super_map, ' ', 6);
LURNZDATA(:,c.dist_super) = dist_super_map.data(iNonSea);
clear('dist_super_map')

rc_map    = importdata(rc_map, ' ', 6);
LURNZDATA(:,c.rc)    = rc_map.data(iNonSea);
clear('rc_map')

ta_map    = importdata(ta_map, ' ', 6);
LURNZDATA(:,c.ta)    = ta_map.data(iNonSea);
clear('ta_map')

LUCAS_map = importdata(LUCAS_map, ' ', 6);
LURNZDATA(:,c.LUCAS) = LUCAS_map.data(iNonSea);
clear('LUCAS_map')

ForestAge_map = importdata(ForestAge_map, ' ', 6);
LURNZDATA(:,c.forestAge) = ForestAge_map.data(iNonSea);
clear('ForestAge_map')

% inform user
msg = sprintf(' - - LUCM0 : maps combined');
disp(msg)

%% Clean maps

% Assume all land pixels with unspecified ownership are not Maori land.
iNonMaori = LURNZDATA(:,c.maori) ~=1;
LURNZDATA(iNonMaori,c.maori) = 0;

% For Forestry Age Map: Two cases of possible mis-match
% We edit the age map to match the 2008 map
% 1 : base_map = not forestry AND age_map = age
iFN = LURNZDATA(:,c.base_map) ~= 3 & LURNZDATA(:,c.forestAge) ~= -9999;
% remove forestry age
LURNZDATA(iFN,c.forestAge) = -9999;

% 2 : base_map = forestry AND age_map = no age
iNF = LURNZDATA(:,c.base_map) == 3 & LURNZDATA(:,c.forestAge) == -9999;
% assign a random age between 0 and 40
% set seed value
rng('default')
% assign
LURNZDATA(iNF,c.forestAge) = round(40 * random('unif',0,1,[sum(iNF),1]));
% Inform user
if sum(iNF)~= 0
    msg = sprintf('WARNING: %d forestry pixels have been assigned random ages',sum(iNF));
    disp(msg)
end

% The LUC map may report values from 9 to 13
% These are numeric codes for lakes, rivers, estuaries, and towns
iNonLUC = LURNZDATA(:,c.luc) > 8;
LURNZDATA(iNonLUC,c.luc) = -9999;

% This appears to be all the cleaning necessary at 25ha resolution.
% No maps contain NaN or NA values.

% remove iNonSea and iNonMaori to save memory space
clear('iNonSea','iNonMaori','iNF','iFN','iNonLUC')

%% Load coefficients by Timar

% target file
coefficients_file = [output_path,'pixel coefficients Timar 2012.xls'];
% Coefficients updates by Timar 27-03-2012

% load data
[coefficients, ~, ~] = xlsread(coefficients_file);

% Rows = 
% 'Slope'
% 'LUC class'
% 'Pasture productivity class'
% 'Distance from nearest town'
% 'Distance from nearest port'
% 'Land is Maori owned'
% 'Constant'

% Columns =
% ['Dairy' , 'Sheep & Beef' , 'Forestry']

%% Preparation for calculating pixel utilities & probabilities

[LURNZDATA, c, iPixels ] = PrepUtilitiesProbabilities(LURNZDATA,c);

%% Calculate pixel utilities

LURNZDATA = CalculatePixelUtilities(LURNZDATA, c, iPixels, coefficients);

%% Convert utilities to probabilities

LURNZDATA = CalculatePixelProbabilities(LURNZDATA , c, iPixels);

%% Remove unneeded columns

% Remove coefficient inputs columns and uTotal
LURNZDATA(:,c.uTotal) = [];
LURNZDATA(:,c.luc:c.dist_super) = [];

% set new column header
clear('c')
c.sorter = 1;
c.LUType = 2;
c.rc = 3;
c.ta = 4;
c.LUCAS = 5;
c.forestAge = 6;
c.uDairy  = 7;
c.uSheep  = 8;
c.uForest = 9;
c.uScrub  = 10;
c.newScrub = 11;

%% Create age vector of new scrub land
%
% We will assume that all scrub starts at age one (it is in its first year
% of growth).
% This means that analysis of scrub sequestration should focus only on that
% scrub that has been planted / begun to regenerate post 2008.

% Make Scrub age vector
ScrubAge_map = -9999 * ones(size(LURNZDATA,1),1);
iScrub = LURNZDATA(:,c.LUType) == 4;
ScrubAge_map(iScrub) = 1;
% insert into LURNZDATA
LURNZDATA(:,c.newScrub) = ScrubAge_map;

%% Calculate the total amount of endogenous rural land

total_endog_area = sum(LURNZDATA(:,c.LUType) == 1 | LURNZDATA(:,c.LUType) == 2 | LURNZDATA(:,c.LUType) == 3 | LURNZDATA(:,c.LUType) == 4);
total_endog_area = total_endog_area * resolution;

%% Save LURNZ GIS DATA information to excel file
% note that LUAM loads LURNZ GIS DATA from the Matlab data file
% But we used to save as an excel file too for historic reasons. 
% 
% % we use dlmwrite as csvwrite is less accurate
% % e.g. 1234567 is recorded as 1234500

%%  Save Matlab output

% remove unwanted files
clear('coeff*', 'column_names', 'file_name', 'iPixels', 'map_path', ...
      'ncol', 'nrow', 'num_maps', 'load_folder', 'sea_pixel', 'msg');

% save output
if resolution == 25
    save([output_path,'LUAM_input_data_',num2str(resolution),'ha'])
elseif resolution == 1
    save([output_path,'LUAM_input_data_',num2str(resolution),'ha'],'-v7.3')
end

%% Inform User saving land use change is complete

msg = sprintf(' - - LUCM0 : setup data complete');
disp(msg)

end

% Subfunction : Construct GIS Header info from ASCI files

function GISheader = ConstructGISHeader(textdata, output_path, resolution )

% Map_GISHeader = [Longitude;Latitude;Xll;Yll;CellSize;No_Value]
% We construct the GISheader from map.textdata
% map.text data should be identical for all maps by construction

Longitude = sscanf(char(textdata(1)),'%*s %f');
Latitude  = sscanf(char(textdata(2)),'%*s %f');
Xll = sscanf(char(textdata(3)),'%*s %f');
Yll = sscanf(char(textdata(4)),'%*s %f');
CellSize  = sscanf(char(textdata(5)),'%*s %f');
No_Value  = sscanf(char(textdata(6)),'%*s %f');

% map.textdata contains the GISheader info
% char converts it from a cell array to a character string
% sscanf breaks the string into its component pieces

GISheader = [Longitude; Latitude; Xll; Yll; CellSize; No_Value];

%% Save GIS header information to excel file
% note that LUAM loads GISheader from the Matlab data file

% file and sheet name
file_name = ['GIS information - ',num2str(resolution),'ha.xls'];
sheet_name = 'Sheet1';

% header labels
GIS_labels = {'ncols'; 'nrows'; 'xllcorner'; 'yllcorner'; ...
              'cellsize'; 'NODATA_value'};

% write to Excel
xlswrite([output_path,file_name],GIS_labels,sheet_name,'A1');
xlswrite([output_path,file_name],GISheader ,sheet_name,'B1');

end

%% Subfunction : Preparation for calculating pixel utilities & probabilities

function [LURNZDATA, c, iPixels ] = PrepUtilitiesProbabilities(LURNZDATA,c)

% Add new columns to c
c.uDairy  = size(LURNZDATA,2) + 1;
c.uSheep  = size(LURNZDATA,2) + 2;
c.uForest = size(LURNZDATA,2) + 3;
c.uScrub  = size(LURNZDATA,2) + 4;
c.uTotal  = size(LURNZDATA,2) + 5;
% Add new columns to LURNZDATA
LURNZDATA(:,c.uDairy:c.uTotal) = 0;

%% identify all endogenous pixels with complete data
iPixels = LURNZDATA(:,c.base_map) <= 4 & ...
          LURNZDATA(:,c.base_map) >= 1 & ...
          ~any( LURNZDATA(:,c.luc:c.dist_super) == -9999 ,2);

% Rescale explanitory variable columns:
%  pasture productivity / 1000
LURNZDATA(:,c.pprod) = LURNZDATA(:,c.pprod) * 0.001;
% distances / 10000
LURNZDATA(:,c.dist_port)  = LURNZDATA(:,c.dist_port)  * 0.0001;
LURNZDATA(:,c.dist_super) = LURNZDATA(:,c.dist_super) * 0.0001;

end

%% Subfunction : Calculate pixel utilities

function [LURNZDATA ] = CalculatePixelUtilities(LURNZDATA,c,iPixels,coeff)

% Temporary fill c.uTotal column with ones for constant in utility calc
LURNZDATA(:,c.uTotal) = 1;

% columns of interest
in_c = [c.uDairy,c.uSheep,c.uForest];
out_c = [c.slope,c.luc,c.pprod,c.dist_super,c.dist_port,c.maori,c.uTotal];
% matrix dimensions: (nx7) x (7x3) = (nx3)

% calculate
LURNZDATA(iPixels,in_c) = LURNZDATA(iPixels,out_c) * coeff;

end

%% Subfunction : Convert utilities to probabilities

function [LURNZDATA ] = CalculatePixelProbabilities(LURNZDATA , c, iPixels)

% Logit model: prob = exp(utility) / sum( exp(utility) )

% columns of interest
in_c = [c.uDairy,c.uSheep,c.uForest,c.uScrub];

% take exponentials
LURNZDATA(iPixels,in_c) = exp( LURNZDATA(iPixels,in_c) );

% uTotal
LURNZDATA(iPixels,c.uTotal) = sum( LURNZDATA(iPixels,in_c) ,2);

% divide through by uTotal
LURNZDATA(iPixels,in_c) = LURNZDATA(iPixels,in_c) ./ ...
                        ( LURNZDATA(iPixels,c.uTotal) * ones(1,4) );

end

